// Helper function to get YouTube cookies
async function getYouTubeCookies() {
    try {
        const response = await new Promise((resolve, reject) => {
            const timeoutId = setTimeout(() => {
                reject(new Error('Timeout waiting for cookies'));
            }, 5000);

            chrome.runtime.sendMessage(
                { action: "getYouTubeCookies" },
                (response) => {
                    clearTimeout(timeoutId);
                    if (chrome.runtime.lastError) {
                        console.error('Runtime error:', chrome.runtime.lastError);
                        reject(new Error(chrome.runtime.lastError.message));
                        return;
                    }
                    console.log('Received response:', response);
                    resolve(response);
                }
            );
        });

        if (!response || !response.success) {
            throw new Error(response?.error || 'Failed to get cookies');
        }

        return response.cookiesText;
    } catch (error) {
        console.error('Error getting cookies:', error);
        return null;
    }
}

console.log('MP3.Tools Extension loaded');

window.addEventListener('load', function() {
    console.log('Window loaded, checking for form...');
    
    const form = document.getElementById('convert-form');
    if (form) {
        console.log('Found convert form');
        
        // Add extension indicator
        const indicator = document.createElement('div');
        indicator.style.cssText = `
            color: green;
            font-size: 12px;
            margin: 10px 0;
            padding: 5px;
            background: #e8f5e9;
            border-radius: 4px;
            text-align: center;
        `;
        indicator.textContent = '✓ MP3.Tools Extension Active (will use your YouTube cookies)';
        indicator.dataset.extensionActive = 'true';
        form.parentNode.insertBefore(indicator, form);

        // Remove any existing event listeners
        const formClone = form.cloneNode(true);
        form.parentNode.replaceChild(formClone, form);
        
        // Get the new form reference
        const newForm = document.getElementById('convert-form');

        // Our new form handler
        newForm.addEventListener('submit', async function(e) {
            e.preventDefault();
            console.log('Form submitted');
            
            // Clear any existing progress tracking
            if (window.progressInterval) {
                clearInterval(window.progressInterval);
                window.progressInterval = null;
            }
            
            // Setup UI elements
            const progressContainer = document.getElementById('progress-container');
            const progressBar = document.getElementById('conversion-progress');
            const progressText = document.getElementById('progress-text');
            const convertButton = document.getElementById('convert-button');
            const videoUrlInput = document.getElementById('video_url');
            
            try {
                // Update UI
                progressContainer.classList.remove('d-none');
                progressBar.style.width = '0%';
                progressBar.setAttribute('aria-valuenow', '0');
                progressBar.textContent = '0%';
                progressText.textContent = 'Preparing conversion...';
                convertButton.disabled = true;
                convertButton.innerHTML = '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Converting...';
                videoUrlInput.readOnly = true;
                
                // Get cookies
                console.log('Requesting YouTube cookies...');
                const cookiesResponse = await getYouTubeCookies();
                
                // Create form data
                const formData = new FormData(newForm);
                if (cookiesResponse) {
                    formData.append('youtube_cookies', cookiesResponse);
                    console.log('Added cookies to request');
                }

                // Start progress tracking
                startProgressTracking(progressBar, progressText);
                
                // Send the conversion request
                const response = await fetch(newForm.action, {
                    method: 'POST',
                    body: formData
                });

                // If we get here, conversion was successful
                console.log('Conversion successful, reloading page...');
                window.location.reload();

            } catch (error) {
                console.error('Error:', error);
                progressContainer.classList.add('d-none');
                convertButton.disabled = false;
                convertButton.innerHTML = 'Convert';
                videoUrlInput.readOnly = false;
                alert('Conversion failed: ' + error.message);
            } finally {
                if (window.progressInterval) {
                    clearInterval(window.progressInterval);
                    window.progressInterval = null;
                }
            }
        });
    }
});

function startProgressTracking(progressBar, progressText) {
    const startTime = Date.now();
    const baseTime = 15000; // Base time of 15 seconds
    const updateInterval = 100; // Update every 100ms
    
    window.progressInterval = setInterval(() => {
        const elapsedTime = Date.now() - startTime;
        let progress;
        
        if (elapsedTime < baseTime) {
            // First 15 seconds: Progress from 0-80%
            progress = (elapsedTime / baseTime) * 80;
        } else {
            // After 15 seconds: Slowly progress from 80-95%
            const remaining = Math.min((elapsedTime - baseTime) / 30000, 1);
            progress = 80 + (remaining * 15);
        }
        
        // Cap at 95%
        progress = Math.min(progress, 95);
        
        progressBar.style.width = progress + '%';
        progressBar.setAttribute('aria-valuenow', progress);
        progressBar.textContent = Math.round(progress) + '%';
        
        progressText.textContent = 'Converting... Please wait.';
    }, updateInterval);
}